
/*! auth.patch.js — Non-destructive Auth Addon (ESM import) — 2025-10-22 */
(function(){
  const FIREBASE_V = "10.14.1";
  const urls = {
    app:  `https://www.gstatic.com/firebasejs/${FIREBASE_V}/firebase-app.js`,
    auth: `https://www.gstatic.com/firebasejs/${FIREBASE_V}/firebase-auth.js`
  };

  let AUTH = null;
  let CURRENT_USER = null;
  let modApp = null, modAuth = null;

  function $(sel, root=document){ return root.querySelector(sel); }
  function on(el, ev, fn, opts){ el && el.addEventListener(ev, fn, opts||false); }
  function toast(kind, msg){
    try { if (typeof window.showToast === "function") return window.showToast(kind, msg); } catch {}
    try { if (kind === "error") console.warn(msg); else console.log(msg); } catch {}
  }

  async function ensureModAuth(){
    if (AUTH) return AUTH;
    if (!modApp)  modApp  = await import(urls.app);
    if (!modAuth) modAuth = await import(urls.auth);
    const { getApp, getApps } = modApp;
    const { getAuth } = modAuth;
    let appInstance; try { appInstance = getApp(); } catch { appInstance = (getApps && getApps()[0]); }
    if (!appInstance) throw new Error("No Firebase app found — ensure firebase.js runs first.");
    AUTH = getAuth(appInstance);
    return AUTH;
  }

  function ensureAuthControls(){
    const pop = $("#settingsPop"); if (!pop) return;
    let box = $("#accountBox", pop);
    if (!box){
      box = document.createElement("section");
      box.id = "accountBox"; box.style.marginTop = "8px";
      box.innerHTML = [
        '<h3 style="margin:0 0 6px">الحساب</h3>',
        '<div id="accountInfo" style="font-size:.95rem;opacity:.9;margin-bottom:.5rem;">لم تُسجّل دخولًا (ضيف)</div>',
        '<div class="row" style="display:flex;gap:.5rem;flex-wrap:wrap">',
        '  <button data-action="login-google">🔑 Google</button>',
        '  <button data-action="login-email">🔑 بريد/كلمة مرور</button>',
        '  <button data-action="logout" style="display:none">🚪 تسجيل الخروج</button>',
        '  <button data-action="sync-now" style="display:none">☁️ مزامنة الآن</button>',
        '</div>',
        '<small style="display:block;margin-top:.5rem;color:#888">البيانات محليًا دائمًا؛ المزامنة بعد تسجيل الدخول.</small>'
      ].join("");
      pop.appendChild(box);
    }
  }

  function updateAccountUI(user){
    const pop = $("#settingsPop"); if (!pop) return;
    const info = $("#accountInfo", pop);
    const g = $('[data-action="login-google"]', pop);
    const e = $('[data-action="login-email"]', pop);
    const lo = $('[data-action="logout"]', pop);
    const sn = $('[data-action="sync-now"]', pop);
    if(!info||!g||!e||!lo||!sn) return;
    if (user && !user.isAnonymous){
      const name = user.displayName || user.email || "حساب";
      info.textContent = `مسجّل: ${name} (uid: ${String(user.uid).slice(0,6)}…)`;
      g.style.display = "none"; e.style.display = "none"; lo.style.display = ""; sn.style.display = "";
    } else {
      info.textContent = "لم تُسجّل دخولًا (ضيف)";
      g.style.display = ""; e.style.display = ""; lo.style.display = "none"; sn.style.display = "none";
    }
  }

  function bindAuthDelegation(){
    const pop = $("#settingsPop"); if (!pop || pop.__patchBoundV2) return; pop.__patchBoundV2 = True = true;
    on(pop, "click", async (ev)=>{
      const btn = ev.target.closest("[data-action]"); if(!btn) return;
      ev.preventDefault();
      const action = btn.getAttribute("data-action");
      try{
        const auth = await ensureModAuth();
        const {
          GoogleAuthProvider, signInWithPopup, signInWithRedirect,
          signInWithEmailAndPassword, createUserWithEmailAndPassword, signOut,
          onAuthStateChanged, getRedirectResult
        } = await import(urls.auth);

        if (action === "login-google"){
          const provider = new GoogleAuthProvider();
          try { await signInWithPopup(auth, provider); }
          catch { await signInWithRedirect(auth, provider); }
        } else if (action === "login-email"){
          const email = prompt("أدخل البريد الإلكتروني:"); if(!email) return;
          const pass = prompt("أدخل كلمة المرور:"); if(!pass) return;
          try { await signInWithEmailAndPassword(auth, email, pass); }
          catch(err){
            if (String(err?.code||"").includes("auth/user-not-found")){
              await createUserWithEmailAndPassword(auth, email, pass);
            } else { throw err; }
          }
        } else if (action === "logout"){
          await signOut(auth); toast("success","تم تسجيل الخروج");
        } else if (action === "sync-now"){
          try { document.dispatchEvent(new CustomEvent("ideo-sync-now")); } catch {}
          toast("success","تم طلب المزامنة الآن");
        }
        updateAccountUI(auth.currentUser || null);

      }catch(err){
        console.warn("Auth action error:", err);
        toast("error","تعذر تنفيذ العملية: " + (err?.message || err));
      }
    });
  }

  function patchSettingsButton(){
    const btn = document.getElementById("settingsBtn") ||
                document.querySelector("[data-settings-btn], .settings-btn, #btnSettings");
    const pop = document.getElementById("settingsPop");
    if (!btn || !pop) return;
    if (btn.__patchSettingsV2) return;
    btn.__patchSettingsV2 = true;
    on(btn, "click", (e)=>{
      e.stopPropagation();
      ensureAuthControls();
      updateAccountUI(CURRENT_USER);
      bindAuthDelegation();
    });
  }

  function observeSettingsIfMissing(){
    if ($("#settingsPop")) return;
    const mo = new MutationObserver(()=>{
      const pop = $("#settingsPop");
      if (pop){
        ensureAuthControls();
        updateAccountUI(CURRENT_USER);
        bindAuthDelegation();
        patchSettingsButton();
        mo.disconnect();
      }
    });
    mo.observe(document.documentElement, {childList:true, subtree:true});
  }

  async function boot(){
    try{
      if (window.FirebaseAPI && typeof window.FirebaseAPI.ready?.then==="function"){
        await window.FirebaseAPI.ready;
      }
    }catch{}

    try{
      const auth = await ensureModAuth();
      const { onAuthStateChanged, getRedirectResult } = await import(urls.auth);
      onAuthStateChanged(auth, (user)=>{
        CURRENT_USER = user || null;
        updateAccountUI(CURRENT_USER);
      });
      try { await getRedirectResult(auth); } catch(e){}
    }catch(e){
      console.warn("Auth patch init warning:", e);
    }

    ensureAuthControls();
    bindAuthDelegation();
    patchSettingsButton();
    observeSettingsIfMissing();
  }

  if (document.readyState === "loading"){
    document.addEventListener("DOMContentLoaded", boot, {once:true});
  } else {
    boot();
  }
})();
