(function(){
  const SAJiL = window.SAJIL = window.SAJIL || {};
  SAJiL.features = SAJiL.features || {};
  const F = SAJiL.features.dashboardFull = {};

  const $ = (id)=>document.getElementById(id);

  function dayKey(d){
    return (SAJiL.util?.dayKey ? SAJiL.util.dayKey(d) : d.toISOString().slice(0,10));
  }

  function sumRange(fromMs, toMs){
    const all = SAJiL.storage.getTx();
    let inc=0, exp=0;
    all.forEach(t=>{
      if(!t || !t.date) return;
      if(t.currency!=='₪') return;
      const ms=new Date(t.date).getTime();
      if(ms<fromMs || ms>toMs) return;
      if(t.type==='income') inc += (+t.amount||0);
      else exp += (+t.amount||0);
    });
    return {inc:Math.round(inc), exp:Math.round(exp), net:Math.round(inc-exp)};
  }

  function drawDonut(canvas, income, expense){
    const ctx = canvas.getContext('2d');
    const W=canvas.width, H=canvas.height;
    ctx.clearRect(0,0,W,H);

    const total = Math.max(1, income+expense);
    const cx=W/2, cy=H/2;
    const r=Math.min(W,H)*0.39;
    const lw=Math.max(18, r*0.48);

    ctx.lineWidth = lw;
    ctx.lineCap = 'round';

    ctx.strokeStyle='rgba(14,27,61,.08)';
    ctx.beginPath(); ctx.arc(cx,cy,r,0,Math.PI*2); ctx.stroke();

    const expA=(expense/total)*Math.PI*2;
    ctx.strokeStyle='rgba(255,77,109,.92)';
    ctx.beginPath(); ctx.arc(cx,cy,r,-Math.PI/2,-Math.PI/2+expA); ctx.stroke();

    const incA=(income/total)*Math.PI*2;
    ctx.strokeStyle='rgba(25,195,125,.92)';
    ctx.beginPath(); ctx.arc(cx,cy,r,-Math.PI/2+expA,-Math.PI/2+expA+incA); ctx.stroke();
  }

  function drawBars(canvas, series, labelMode='day'){
    const ctx=canvas.getContext('2d');
    const parent = canvas.parentElement;
    const W = Math.max(280, parent.clientWidth);
    if(canvas.width !== W) canvas.width = W;
    const H = canvas.height;

    ctx.clearRect(0,0,W,H);

    const maxV = Math.max(1, ...series.map(d=>Math.max(d.inc,d.exp)));
    const pad=10;
    const colW=(W-pad*2)/series.length;
    const base=H-18;

    ctx.font='900 10px Cairo, system-ui';
    ctx.textAlign='center';

    series.forEach((d,i)=>{
      const x = pad + i*colW + colW/2;
      const hInc = (d.inc/maxV)*(H-30);
      const hExp = (d.exp/maxV)*(H-30);

      ctx.fillStyle='rgba(255,77,109,.55)';
      ctx.fillRect(x-colW*0.18, base-hExp, colW*0.16, hExp);

      ctx.fillStyle='rgba(25,195,125,.55)';
      ctx.fillRect(x+colW*0.02, base-hInc, colW*0.16, hInc);

      let lab = d.key.slice(8,10);
      if(labelMode==='month') lab = d.key; // "01".."12"
      ctx.fillStyle='rgba(14,27,61,.55)';
      ctx.fillText(lab, x, H-6);
    });
  }

  function topExpenseDays(fromMs, toMs, limit=3){
    const all = SAJiL.storage.getTx();
    const m = {};
    all.forEach(t=>{
      if(!t || !t.date) return;
      if(t.currency!=='₪') return;
      const ms=new Date(t.date).getTime();
      if(ms<fromMs || ms>toMs) return;
      const key = (t.date||'').slice(0,10);
      if(!m[key]) m[key] = {inc:0, exp:0};
      if(t.type==='income') m[key].inc += (+t.amount||0);
      else m[key].exp += (+t.amount||0);
    });

    const arr = Object.entries(m).map(([k,v])=>({key:k, inc:Math.round(v.inc), exp:Math.round(v.exp), net:Math.round(v.inc-v.exp)}));
    arr.sort((a,b)=>b.exp-a.exp);
    return arr.slice(0,limit);
  }

  function buildWeekSeries(){
    const now=new Date();
    const out=[];
    for(let i=6;i>=0;i--){
      const d=new Date(now); d.setDate(d.getDate()-i);
      const k=dayKey(d);
      const sums = sumRange(
        new Date(k+'T00:00:00').getTime(),
        new Date(k+'T23:59:59').getTime()
      );
      out.push({key:k, inc:sums.inc, exp:sums.exp});
    }
    return out;
  }

  function buildMonthSeries(){
    // آخر 12 شهر (مختصر وجميل)
    const now=new Date();
    const out=[];
    for(let i=11;i>=0;i--){
      const d=new Date(now.getFullYear(), now.getMonth()-i, 1);
      const from=new Date(d.getFullYear(), d.getMonth(), 1, 0,0,0,0).getTime();
      const to=new Date(d.getFullYear(), d.getMonth()+1, 0, 23,59,59,999).getTime();
      const sums = sumRange(from,to);
      const mm = String(d.getMonth()+1).padStart(2,'0');
      out.push({key:mm, inc:sums.inc, exp:sums.exp, from, to});
    }
    return out;
  }

  function rangeLabel(fromMs, toMs){
    const a = new Date(fromMs).toISOString().slice(0,10);
    const b = new Date(toMs).toISOString().slice(0,10);
    return `${a} → ${b}`;
  }

  F.state = { mode:'week' };

  F.render = function(){
    const sheet = $('dashSheet');
    if(!sheet || sheet.classList.contains('hidden')) return;

    const mode = F.state.mode;

    let fromMs, toMs, series;
    if(mode==='week'){
      series = buildWeekSeries();
      fromMs = new Date(series[0].key+'T00:00:00').getTime();
      toMs   = new Date(series[6].key+'T23:59:59').getTime();
      $('dashModeWeek')?.classList.add('active');
      $('dashModeMonth')?.classList.remove('active');
    }else{
      const m = buildMonthSeries();
      series = m;
      fromMs = m[0].from;
      toMs   = m[m.length-1].to;
      $('dashModeMonth')?.classList.add('active');
      $('dashModeWeek')?.classList.remove('active');
    }

    $('dashRangeLabel').textContent = rangeLabel(fromMs,toMs);

    const sums = sumRange(fromMs,toMs);
    $('dashIncBig').textContent = `${sums.inc}₪`;
    $('dashExpBig').textContent = `${sums.exp}₪`;
    $('dashNetBig').textContent = `${sums.net}₪`;

    drawDonut($('dashDonutBig'), sums.inc, sums.exp);

    const barsCanvas = $('dashBarsBig');
    drawBars(barsCanvas, series, mode==='month'?'month':'day');

    $('dashBarsHint').textContent = mode==='week'
      ? 'كل يوم: عمودان (مصروف أحمر، مدخول أخضر)'
      : 'كل شهر: عمودان (مصروف أحمر، مدخول أخضر)';

    // Top expense days (only week range)
    const top = topExpenseDays(fromMs,toMs,3);
    const box = $('dashTopDays');
    box.innerHTML = '';
    top.forEach(d=>{
      const div = document.createElement('div');
      div.className = 'dashDay';
      div.innerHTML = `<div><b>${d.key}</b><div class="muted">مصروف ${d.exp}₪ • مدخول ${d.inc}₪</div></div><b>${d.net}₪</b>`;
      div.addEventListener('click', ()=>{
        // Filter UI to that day quickly
        SAJiL.state.period='custom';
        SAJiL.ui.fromDate().value = d.key;
        SAJiL.ui.toDate().value = d.key;
        SAJiL.state.customFrom = new Date(d.key+'T00:00:00');
        SAJiL.state.customTo   = new Date(d.key+'T23:59:59');
        SAJiL.render();
        // close sheet
        $('dashSheet').classList.add('hidden');
        SAJiL.ui.modalBackdrop().classList.add('hidden');
      });
      box.appendChild(div);
    });
  };

  F.init = function(){
    // open/close
    $('asDash')?.addEventListener('click', ()=>{
      SAJiL.ui.modalBackdrop().classList.remove('hidden');
      $('dashSheet').classList.remove('hidden');
      F.render();
    });
    $('closeDash')?.addEventListener('click', ()=>{
      $('dashSheet').classList.add('hidden');
      SAJiL.ui.modalBackdrop().classList.add('hidden');
    });

    $('dashModeWeek')?.addEventListener('click', ()=>{
      F.state.mode='week';
      F.render();
    });
    $('dashModeMonth')?.addEventListener('click', ()=>{
      F.state.mode='month';
      F.render();
    });

    // re-render on resize
    window.addEventListener('resize', ()=>{
      clearTimeout(F._t);
      F._t=setTimeout(F.render, 120);
    });
  };
})();
